<?php

/**
 * Plugin Name: Fastland Helper
 * Description: This plugin is compatible with Fastland WordPress Landing Page Theme
 * Plugin URI: http://FinestDevs.com/
 * Version:     1.3.0
 * Author:      FinestDevs
 * Author URI:  https://FinestDevs.com
 * Text Domain: fastland-hp
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

/**
 * Main fastland-hp Class
 *
 * The main class that initiates and runs the plugin.
 *
 * @since 1.0.0
 */
final class fastland_elementor
{

    /**
     * Plugin Version
     *
     * @since 1.0.0
     *
     * @var string The plugin version.
     */
    const VERSION = '1.2.0';

    /**
     * Minimum Elementor Version
     *
     * @since 1.0.0
     *
     * @var string Minimum Elementor version required to run the plugin.
     */
    const MINIMUM_ELEMENTOR_VERSION = '2.0.0';

    /**
     * Minimum PHP Version
     *
     * @since 1.0.0
     *
     * @var string Minimum PHP version required to run the plugin.
     */
    const MINIMUM_PHP_VERSION = '5.6';

    /**
     * Instance
     *
     * @since 1.0.0
     *
     * @access private
     * @static
     *
     * @var fastland_elementor The single instance of the class.
     */
    private static $_instance = null;

    /**
     * Instance
     *
     * Ensures only one instance of the class is loaded or can be loaded.
     *
     * @since 1.0.0
     *
     * @access public
     * @static
     *
     * @return fastland_elementor An instance of the class.
     */
    public static function instance()
    {
        if (is_null(self::$_instance)) {
            self::$_instance = new self();
        }
        return self::$_instance;
    }

    /**
     * Constructor
     *
     * @since 1.0.0
     *
     * @access public
     */
    public function __construct()
    {
        add_action('init', [$this, 'i18n']);
        add_action('plugins_loaded', [$this, 'init']);
    }

    /**
     * Load Textdomain
     *
     * Load plugin localization files.
     *
     * Fired by `init` action hook.
     *
     * @since 1.0.0
     *
     * @access public
     */
    public function i18n()
    {
        load_plugin_textdomain('fastland-hp');
    }

    /**
     * Initialize the plugin
     *
     * Load the plugin only after Elementor (and other plugins) are loaded.
     * Checks for basic plugin requirements, if one check fail don't continue,
     * if all check have passed load the files required to run the plugin.
     *
     * Fired by `plugins_loaded` action hook.
     *
     * @since 1.0.0
     *
     * @access public
     */
    public function init()
    {

        // $this->update_checker();

        // Require the main plugin file
        require __DIR__ . '/inc/helper-functions.php';
        require_once __DIR__ . '/inc/custom-post-types.php';
        require_once __DIR__ . '/inc/shortcodes.php';
        require __DIR__ . '/inc/elmentor-extender.php';
        require __DIR__ . '/inc/Fastland_Nav_Walker.php';
        require __DIR__ . '/inc/admin-column.php';

        // Check if Elementor installed and activated
        if (!did_action('elementor/loaded')) {
            add_action('admin_notices', [$this, 'admin_notice_missing_main_plugin']);
            return;
        }

        // Check for required Elementor version
        if (!version_compare(ELEMENTOR_VERSION, self::MINIMUM_ELEMENTOR_VERSION, '>=')) {
            add_action('admin_notices', [$this, 'admin_notice_minimum_elementor_version']);
            return;
        }

        // Check for required PHP version
        if (version_compare(PHP_VERSION, self::MINIMUM_PHP_VERSION, '<')) {
            add_action('admin_notices', [$this, 'admin_notice_minimum_php_version']);
            return;
        }

        // Add Plugin actions
        add_action('elementor/widgets/register', [$this, 'init_widgets']);
        add_action('elementor/widgets/register', [$this, 'widget_styles']);
        add_action('elementor/frontend/after_enqueue_styles', [$this, 'widget_styles']);
        add_action('elementor/frontend/after_register_scripts', [$this, 'widget_scripts']);
    }

    /* Plugin update checker */
    public function update_checker()
    {
        if (!class_exists('Puc_v4_Factory')) {
            require_once __DIR__ . '/lib/plugin-update-checker/plugin-update-checker.php';
        }
        $MyUpdateChecker = Puc_v4_Factory::buildUpdateChecker(
            'https://update.grayic.com?action=get_metadata&slug=fastland-helper', //Metadata URL.
            __FILE__, //Full path to the main plugin file.
            'fastland-helper'
        );
    }

    /**
     * Admin notice
     *
     * Warning when the site doesn't have Elementor installed or activated.
     *
     * @since 1.0.0
     *
     * @access public
     */
    public function admin_notice_missing_main_plugin()
    {
        if (isset($_GET['activate'])) {
            unset($_GET['activate']);
        }

        $message = sprintf(
            /* translators: 1: Plugin name 2: Elementor */
            esc_html__('"%1$s" requires "%2$s" to be installed and activated.', 'fastland-hp'),
            '<strong>' . esc_html__('fastland-hp', 'fastland-hp') . '</strong>',
            '<strong>' . esc_html__('Elementor', 'fastland-hp') . '</strong>'
        );

        printf('<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message);
    }

    /**
     * Admin notice
     *
     * Warning when the site doesn't have a minimum required Elementor version.
     *
     * @since 1.0.0
     *
     * @access public
     */
    public function admin_notice_minimum_elementor_version()
    {
        if (isset($_GET['activate'])) {
            unset($_GET['activate']);
        }

        $message = sprintf(
            /* translators: 1: Plugin name 2: Elementor 3: Required Elementor version */
            esc_html__('"%1$s" requires "%2$s" version %3$s or greater.', 'fastland-hp'),
            '<strong>' . esc_html__('Fastland Helper', 'fastland-hp') . '</strong>',
            '<strong>' . esc_html__('Elementor', 'fastland-hp') . '</strong>',
            self::MINIMUM_ELEMENTOR_VERSION
        );

        printf('<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message);
    }

    /**
     * Admin notice
     *
     * Warning when the site doesn't have a minimum required PHP version.
     *
     * @since 1.0.0
     *
     * @access public
     */
    public function admin_notice_minimum_php_version()
    {
        if (isset($_GET['activate'])) {
            unset($_GET['activate']);
        }

        $message = sprintf(
            /* translators: 1: Plugin name 2: PHP 3: Required PHP version */
            esc_html__('"%1$s" requires "%2$s" version %3$s or greater.', 'fastland-hp'),
            '<strong>' . esc_html__('fastland-hp', 'fastland-hp') . '</strong>',
            '<strong>' . esc_html__('PHP', 'fastland-hp') . '</strong>',
            self::MINIMUM_PHP_VERSION
        );

        printf('<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message);
    }

    /**
     * Init Widgets
     *
     * Include widgets files and register them
     *
     * @since 1.0.0
     *
     * @access public
     */
    public function init_widgets($widgets_manager)
    {

        // Include Widget files
        if (fastland_check_cpt('service')) {

            require_once __DIR__ . '/widgets/services.php';

            require_once __DIR__ . '/widgets/service-categories.php';
        }

        if (fastland_check_cpt('job')) {
            require_once __DIR__ . '/widgets/job/job.php';
            require_once __DIR__ . '/widgets/job/job-search.php';
        }

        if (fastland_check_cpt('team')) {
            require_once __DIR__ . '/widgets/team.php';
        }

        if (fastland_check_cpt('testimonial')) {
            require_once __DIR__ . '/widgets/testimonial.php';
        }

        if (fastland_check_cpt('portfolio')) {
            require_once __DIR__ . '/widgets/portfolio/portfolio.php';

            require_once __DIR__ . '/widgets/portfolio/portfolio-gallery.php';

            require_once __DIR__ . '/widgets/portfolio/single-portfolio-meta.php';
        }

        /* Tab */
        require_once __DIR__ . '/widgets/tab.php';

        /* Featured */
        require_once __DIR__ . '/widgets/featured.php';

        /* Testimonail Simple */
        require_once __DIR__ . '/widgets/testimonial-simple.php';

        /*Logo */
        require_once __DIR__ . '/widgets/logo.php';

        /* Nav Menu */
        require_once __DIR__ . '/widgets/main-menu.php';
    }

    public function widget_styles()
    {
        wp_dequeue_style('elementor-animations');
        wp_register_style('fastland-fas-6', '//cdnjs.cloudflare.com/ajax/libs/font-awesome/6.1.1/css/all.min.css', [], '6.1.1');
        wp_register_style('owl-carousel', plugins_url('/assets/css/owl.carousel.min.css', __FILE__), [], microtime());
        wp_enqueue_style('owl-carousel');

        wp_enqueue_style('fastland-elementor-animations', plugins_url('/assets/css/animate.css', __FILE__), [], microtime());
        wp_enqueue_style('slick', plugins_url('/assets/css/slick.css', __FILE__), [], microtime());

        wp_enqueue_style('magnific-popup', plugins_url('/assets/css/magnific-popup.css', __FILE__), [], microtime());
        wp_enqueue_style('fastland-addons', plugins_url('/assets/css/addons.css', __FILE__), ['fd-addons'], microtime());
        wp_style_add_data('fastland-addons', 'rtl', 'replace');
    }

    public function widget_scripts()
    {
        $fastland = get_option('fastland');
        if ('1' !== $fastland['gmap_api']) {
            $api = sprintf('https://maps.googleapis.com/maps/api/js?key=%1$s&language=%2$s', $fastland['gmap_api'], 'en');
            wp_register_script('fastland-maps-api-js', $api, array(), '', false);
        }
        wp_register_script('fastland-maps-js', plugins_url('/assets/js/fastland-maps.js', __FILE__), ['jquery']);

        wp_register_script('owl-carousel', plugins_url('/assets/js/owl.carousel.min.js', __FILE__), ['jquery']);
        wp_enqueue_script('owl-carousel');

        wp_enqueue_script('jquery-ui-datepicker');
        wp_enqueue_script('slick', plugins_url('/assets/js/slick.js', __FILE__), ['jquery']);
        wp_enqueue_script('isotope', plugins_url('/assets/js/isotope.pkgd.min.js', __FILE__), ['jquery']);
        wp_enqueue_script('packery', plugins_url('/assets/js/packery-mode.pkgd.min.js', __FILE__), ['jquery', 'isotope']);
        wp_enqueue_script('imagesloaded', plugins_url('/assets/js/imagesloaded.pkgd.min.js', __FILE__), ['jquery', 'isotope']);
        wp_enqueue_script('magnific-popup', plugins_url('/assets/js/jquery.magnific-popup.min.js', __FILE__), ['jquery'], microtime(), true);
        wp_enqueue_script('fastland-addon', plugins_url('/assets/js/addon.js', __FILE__), ['jquery', 'fd-addons'], microtime(), true);
    }
}

fastland_elementor::instance();

/**
 * adding new category
 */
function fastland_add_elementor_widget_categories($elements_manager)
{
    $elements_manager->add_category(
        'fastland-addons',
        [
            'title' => __('Fastland addons', 'fastland-hp'),
            'icon'  => 'fa fa-plug',
        ]
    );
}
add_action('elementor/elements/categories_registered', 'fastland_add_elementor_widget_categories');
