<?php
if (!defined('ABSPATH')) {
    exit;
}
// Exit if accessed directly
/**
 * Elementor Hello World
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class Fastland_Services extends \Elementor\Widget_Base
{
    /**
     * Retrieve the widget name.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget name.
     */
    public function get_name()
    {
        return 'fastland-service';
    }

    /**
     * Style dependencies
     * 
     */
    public function get_style_depends()
    {
        return ['fastland-fas-6'];
    }

    /**
     * Retrieve the widget title.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget title.
     */
    public function get_title()
    {
        return __('Fastland Services', 'fastland-hp');
    }
    /**
     * Retrieve the widget icon.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget icon.
     */
    public function get_icon()
    {
        return 'eicon-settings';
    }
    /**
     * Retrieve the list of categories the widget belongs to.
     *
     * Used to determine where to display the widget in the editor.
     *
     * Note that currently Elementor supports only one category.
     * When multiple categories passed, Elementor uses the first one.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget categories.
     */
    public function get_categories()
    {
        return ['fastland-addons'];
    }
    /**
     * Register the widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function register_controls()
    {
        $this->start_controls_section(
            'section_query',
            [
                'label' => __('Query', 'fastland-ts'),
            ]
        );
        $this->add_control(
            'posts_per_page',
            [
                'label'   => __('Posts per page', 'fastland-ts'),
                'type'    => \Elementor\Controls_Manager::NUMBER,
                'default' => 5,
            ]
        );
        $this->add_control(
            'source',
            [
                'label'   => __('Source', 'fastland-ts'),
                'type'    => \Elementor\Controls_Manager::SELECT,
                'options' => [
                    'service'          => 'Service',
                    'manual_selection' => 'Manual Selection',
                    'related'          => 'Related',
                ],
                'default' => 'service',
            ]
        );
        $this->add_control(
            'manual_selection',
            [
                'label'       => __('Manual Selection', 'fastland-ts'),
                'type'        => \Elementor\Controls_Manager::SELECT2,
                'description' => __('Get specific template posts', 'fastland-ts'),
                'label_block' => true,
                'multiple'    => true,
                'options'     => fastland_cpt_slug_and_id('service'),
                'default'     => [],
                'condition'   => [
                    'source' => 'manual_selection',
                ],
            ]
        );
        $this->start_controls_tabs(
            'include_exclude_tabs'
        );
        $this->start_controls_tab(
            'include_tabs',
            [
                'label'     => __('Include', 'fastland-ts'),
                'condition' => [
                    'source!' => 'manual_selection',
                ],
            ]
        );
        $this->add_control(
            'include_by',
            [
                'label'       => __('Include by', 'fastland-ts'),
                'type'        => \Elementor\Controls_Manager::SELECT2,
                'label_block' => true,
                'multiple'    => true,
                'options'     => [
                    'tags'     => 'Tags',
                    'category' => 'Category',
                    'author'   => 'Author',
                ],
                'default'     => [],
                'condition'   => [
                    'source!' => 'manual_selection',
                ],
            ]
        );
        $this->add_control(
            'include_categories',
            [
                'label'       => __('Include categories', 'fastland-ts'),
                'type'        => \Elementor\Controls_Manager::SELECT2,
                'description' => __('Get templates for specific category(s)', 'fastland-ts'),
                'label_block' => true,
                'multiple'    => true,
                'options'     => fastland_cpt_taxonomy_slug_and_name('service-category'),
                'default'     => [],
                'condition'   => [
                    'include_by' => 'category',
                    'source!'    => 'related',
                ],
            ]
        );
        $this->add_control(
            'include_tags',
            [
                'label'       => __('Include Tags', 'fastland-ts'),
                'type'        => \Elementor\Controls_Manager::SELECT2,
                'description' => __('Get templates for specific tag(s)', 'fastland-ts'),
                'label_block' => true,
                'multiple'    => true,
                'options'     => fastland_cpt_taxonomy_slug_and_name('service-tag'),
                'default'     => [],
                'condition'   => [
                    'include_by' => 'tags',
                    'source!'    => 'related',
                ],
            ]
        );
        $this->add_control(
            'include_authors',
            [
                'label'       => __('Include authors', 'fastland-ts'),
                'type'        => \Elementor\Controls_Manager::SELECT2,
                'description' => __('Get templates for specific tag(s)', 'fastland-ts'),
                'label_block' => true,
                'multiple'    => true,
                'options'     => fastland_cpt_author_slug_and_id('service'),
                'default'     => [],
                'condition'   => [
                    'include_by' => 'author',
                ],
            ]
        );
        $this->end_controls_tab();
        $this->start_controls_tab(
            'exclude_tabs',
            [
                'label'     => __('Exclude', 'fastland-ts'),
                'condition' => [
                    'source!' => 'manual_selection',
                ],
            ]
        );
        $this->add_control(
            'exclude_by',
            [
                'label'       => __('Exclude by', 'fastland-ts'),
                'type'        => \Elementor\Controls_Manager::SELECT2,
                'label_block' => true,
                'multiple'    => true,
                'options'     => [
                    'tags'         => 'tags',
                    'category'     => 'Category',
                    'author'       => 'Author',
                    'current_post' => 'Current Post',
                ],
                'default'     => [],
                'condition'   => [
                    'source!' => 'manual_selection',
                ],
            ]
        );
        $this->add_control(
            'exclude_categories',
            [
                'label'       => __('Exclude categories', 'fastland-ts'),
                'type'        => \Elementor\Controls_Manager::SELECT2,
                'description' => __('Get templates for specific category(s)', 'fastland-ts'),
                'label_block' => true,
                'multiple'    => true,
                'options'     => fastland_cpt_taxonomy_slug_and_name('service-category'),
                'default'     => [],
                'condition'   => [
                    'exclude_by' => 'category',
                    'source!'    => 'related',
                ],
            ]
        );
        $this->add_control(
            'exclude_tags',
            [
                'label'       => __('Exclude Tags', 'fastland-ts'),
                'type'        => \Elementor\Controls_Manager::SELECT2,
                'description' => __('Get templates for specific tag(s)', 'fastland-ts'),
                'label_block' => true,
                'multiple'    => true,
                'options'     => fastland_cpt_taxonomy_slug_and_name('service-tag'),
                'default'     => [],
                'condition'   => [
                    'exclude_by' => 'tags',
                    'source!'    => 'related',
                ],
            ]
        );
        $this->add_control(
            'exclude_authors',
            [
                'label'       => __('Exclude authors', 'fastland-ts'),
                'type'        => \Elementor\Controls_Manager::SELECT2,
                'description' => __('Get templates for specific tag(s)', 'fastland-ts'),
                'label_block' => true,
                'multiple'    => true,
                'options'     => fastland_cpt_author_slug_and_id('service'),
                'default'     => [],
                'condition'   => [
                    'exclude_by' => 'author',
                ],
            ]
        );
        $this->end_controls_tab();
        $this->end_controls_tabs();
        $this->add_control(
            'orderby',
            [
                'label'   => __('Order By', 'fastland-ts'),
                'type'    => \Elementor\Controls_Manager::SELECT,
                'options' => [
                    'date'       => 'Date',
                    'title'      => 'title',
                    'menu_order' => 'Menu Order',
                    'rand'       => 'Random',
                ],
                'default' => 'date',
            ]
        );
        $this->add_control(
            'order',
            [
                'label'   => __('Order', 'fastland-ts'),
                'type'    => \Elementor\Controls_Manager::SELECT,
                'options' => [
                    'ASC'  => 'ASC',
                    'DESC' => 'DESC',
                ],
                'default' => 'DESC',
            ]
        );
        $this->end_controls_section();
        $this->start_controls_section(
            'section_layout',
            [
                'label' => __('Layout', 'fastland-hp'),
                'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );
        $this->add_responsive_control(
            'post_grid',
            [
                'label'          => __('Post grid', 'fastland-hp'),
                'type'           => \Elementor\Controls_Manager::SELECT,
                'options'        => array(
                    '12' => '1 Column',
                    '6'  => '2 Column',
                    '4'  => '3 Column',
                    '3'  => '4 Column',
                ),
                'default'        => 3,
                'tablet_default' => 6,
                'mobile_default' => 12,
            ]
        );
        $this->add_responsive_control(
            'column_gap',
            [
                'label'     => __('Column Gap', 'fastland-hp'),
                'type'      => \Elementor\Controls_Manager::SLIDER,
                'range'     => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'devices'   => ['desktop', 'tablet', 'mobile'],
                'selectors' => [
                    '{{WRAPPER}} .fastland-service-widget-wrap' => 'padding: 0 {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        $this->add_responsive_control(
            'row_gap',
            [
                'label'     => __('Row Gap', 'fastland-hp'),
                'type'      => \Elementor\Controls_Manager::SLIDER,
                'range'     => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'devices'   => ['desktop', 'tablet', 'mobile'],
                'selectors' => [
                    '{{WRAPPER}} .fastland-service-widget-wrap' => 'margin: 0 0 {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        $this->end_controls_section();
        $this->start_controls_section(
            'section_content',
            [
                'label' => __('Content', 'fastland-hp'),
                'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );
        $this->add_control(
            'show_excerpt',
            [
                'label'        => __('Show Excerpt', 'fastland-hp'),
                'type'         => \Elementor\Controls_Manager::SWITCHER,
                'label_on'     => __('Show', 'fastland-hp'),
                'label_off'    => __('Hide', 'fastland-hp'),
                'return_value' => 'yes',
                'default'      => 'yes',
            ]
        );
        $this->add_control(
            'excerpt_limit',
            [
                'label'    => __('Excerpt Word Limit', 'fastland-hp'),
                'type'     => \Elementor\Controls_Manager::SLIDER,
                'range'    => [
                    'px' => [
                        'min' => 0,
                        'max' => 1000,
                    ],
                ],
                'conditon' => [
                    'show_excerpt' => 'yes',
                ],
            ]
        );
        $this->add_control(
            'title_limit',
            [
                'label' => __('Title Word Limit', 'fastland-hp'),
                'type'  => \Elementor\Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 1000,
                    ],
                ],
            ]
        );
        $this->end_controls_section();
        $this->start_controls_section(
            'section_btn',
            [
                'label' => __('Readmore', 'fastland-hp'),
                'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );
        $this->add_control(
            'show_readmore',
            [
                'label'        => __('Readmore button', 'fastland-hp'),
                'type'         => \Elementor\Controls_Manager::SWITCHER,
                'label_on'     => __('Show', 'fastland-hp'),
                'label_off'    => __('Hide', 'fastland-hp'),
                'return_value' => 'yes',
                'default'      => 'yes',
            ]
        );
        $this->add_control(
            'readmore_text',
            [
                'label'    => __('Readmore text', 'fastland-hp'),
                'type'     => \Elementor\Controls_Manager::TEXT,
                'default'  => __('READ MORE', 'fastland-hp'),
                'conditon' => [
                    'show_readmore' => 'yes',
                ],
            ]
        );
        $this->add_control(
            'icon',
            [
                'label'    => __('Icon', 'fastland-hp'),
                'type'     => \Elementor\Controls_Manager::ICONS,
                'conditon' => [
                    'show_readmore' => 'yes',
                ],
            ]
        );
        $this->add_control(
            'icon_position',
            [
                'label'    => __('Icon Position', 'fastland-hp'),
                'type'     => \Elementor\Controls_Manager::SELECT,
                'default'  => 'after',
                'options'  => [
                    'before' => __('Before', 'fastland-hp'),
                    'after'  => __('After', 'fastland-hp'),
                ],
                'conditon' => [
                    'show_readmore' => 'yes',
                ],
            ]
        );
        $this->add_responsive_control(
            'button_align',
            [
                'label'        => __('Align', 'fastland-hp'),
                'type'         => \Elementor\Controls_Manager::CHOOSE,
                'options'      => [
                    'left'   => [
                        'title' => __('Left', 'fastland-hp'),
                        'icon'  => 'fa fa-align-left',
                    ],
                    'center' => [
                        'title' => __('top', 'fastland-hp'),
                        'icon'  => 'fa fa-align-center',
                    ],
                    'right'  => [
                        'title' => __('Right', 'fastland-hp'),
                        'icon'  => 'fa fa-align-right',
                    ],
                ],
                'devices'      => ['desktop', 'tablet', 'mobile'],
                'prefix_class' => 'content-align%s-',
                'toggle'       => true,
            ]
        );
        $this->end_controls_section();
        $this->start_controls_section(
            'section_icon_style',
            [
                'label' => __('Icon', 'fastland-hp'),
                'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );
        $this->start_controls_tabs(
            'icon_hover_tabs'
        );
        $this->start_controls_tab(
            'icon_normal_tab',
            [
                'label' => __('Normal', 'fastland-hp'),
            ]
        );
        $this->add_control(
            'icon_color',
            [
                'label'     => __('Icon Color', 'fastland-hp'),
                'type'      => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .service-icon i' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'icon_background',
            [
                'label'     => __('Icon Background', 'fastland-hp'),
                'type'      => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .service-icon' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        $this->add_group_control(
            \Elementor\Group_Control_Box_Shadow::get_type(),
            [
                'name'     => 'icon_shadow',
                'label'    => __('Icon Shadow', 'fastland-hp'),
                'selector' => '{{WRAPPER}} .service-icon',
            ]
        );
        $this->add_responsive_control(
            'icon_size',
            [
                'label'     => __('Icon Size', 'fastland-hp'),
                'type'      => \Elementor\Controls_Manager::SLIDER,
                'range'     => [
                    'px' => [
                        'min' => 0,
                        'max' => 1000,
                    ],
                    '%'  => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'devices'   => ['desktop', 'tablet', 'mobile'],
                'selectors' => [
                    '{{WRAPPER}} .service-icon i' => 'font-size: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        $this->add_responsive_control(
            'icon_box_size',
            [
                'label'     => __('Icon Box Size', 'fastland-hp'),
                'type'      => \Elementor\Controls_Manager::SLIDER,
                'range'     => [
                    'px' => [
                        'min' => 0,
                        'max' => 1000,
                    ],
                    '%'  => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'devices'   => ['desktop', 'tablet', 'mobile'],
                'selectors' => [
                    '{{WRAPPER}} .service-icon' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        $this->add_responsive_control(
            'icon_gap',
            [
                'label'      => __('Icon Gap', 'fastland-hp'),
                'type'       => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .service-icon ' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        $this->end_controls_tab();
        $this->start_controls_tab(
            'style_hover_tab',
            [
                'label' => __('Hover', 'fastland-hp'),
            ]
        );
        $this->add_control(
            'icon_hover_color',
            [
                'label'     => __('Icon Color', 'fastland-hp'),
                'type'      => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}}:hover .service-icon i' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'icon_hover_background',
            [
                'label'     => __('Icon Background', 'fastland-hp'),
                'type'      => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}}:hover .service-icon' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        $this->add_group_control(
            \Elementor\Group_Control_Box_Shadow::get_type(),
            [
                'name'     => 'icon_shadow_hover',
                'label'    => __('Icon Shadow', 'fastland-hp'),
                'selector' => '{{WRAPPER}}:hover .service-icon',
            ]
        );
        $this->end_controls_tab();
        $this->end_controls_tabs();
        $this->end_controls_section();
        $this->start_controls_section(
            'content_style',
            [
                'label' => __('Content', 'fastland-hp'),
                'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );
        $this->start_controls_tabs(
            'content_style_tabs'
        );
        $this->start_controls_tab(
            'content_style_normal_tab',
            [
                'label' => __('Normal', 'fastland-hp'),
            ]
        );
        $this->add_control(
            'title_color',
            [
                'label'     => __('Title Color', 'fastland-hp'),
                'type'      => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .fastland-service-widget-item .service-title' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'excerpt_color',
            [
                'label'     => __('Excerpt Color', 'fastland-hp'),
                'type'      => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .fastland-service-widget-item p' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name'     => 'heading_typography',
                'label'    => __('Title Typography', 'fastland-hp'),
                'selector' => '{{WRAPPER}} .fastland-service-widget-item .service-title',
            ]
        );
        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name'     => 'excerpt_typ',
                'label'    => __('Excerpt Typography', 'fastland-hp'),
                'selector' => '{{WRAPPER}} .fastland-service-widget-item p',
            ]
        );
        $this->add_control(
            'title_br',
            [
                'type' => \Elementor\Controls_Manager::DIVIDER,
            ]
        );
        $this->add_responsive_control(
            'title_gap',
            [
                'label'     => __('Title Gap', 'fastland-hp'),
                'type'      => \Elementor\Controls_Manager::SLIDER,
                'range'     => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'devices'   => ['desktop', 'tablet', 'mobile'],
                'selectors' => [
                    '{{WRAPPER}} .fastland-service-widget-item .service-title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        $this->add_responsive_control(
            'title_padding',
            [
                'label'      => __('Title Padding', 'fastland-hp'),
                'type'       => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .fastland-service-widget-item .service-title'          => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    'body.rtl {{WRAPPER}} .fastland-service-widget-item .service-title' => 'padding: {{TOP}}{{UNIT}} {{LEFT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{RIGHT}}{{UNIT}};',
                ],
            ]
        );
        $this->add_responsive_control(
            'content_padding',
            [
                'label'      => __('Content Padding', 'fastland-hp'),
                'type'       => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .fastland-service-widget-item .service-content p'          => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    'body.rtl {{WRAPPER}} .fastland-service-widget-item .service-content p' => 'padding:
                    {{TOP}}{{UNIT}} {{LEFT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{RIGHT}}{{UNIT}};',
                ],
            ]
        );
        $this->add_responsive_control(
            'content_box_padding',
            [
                'label'      => __('Content Box Padding', 'fastland-hp'),
                'type'       => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .fastland-service-widget-item .service-content'          => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    'body.rtl {{WRAPPER}} .fastland-service-widget-item .service-content' => 'padding:
                    {{TOP}}{{UNIT}} {{LEFT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{RIGHT}}{{UNIT}};',
                ],
            ]
        );
        $this->end_controls_tab();
        $this->start_controls_tab(
            'content_style_hover_tab',
            [
                'label' => __('Hover', 'fastland-hp'),
            ]
        );
        $this->add_control(
            'title_hover_color',
            [
                'label'     => __('Title Color', 'fastland-hp'),
                'type'      => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .fastland-service-widget-item:hover .service-title' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'excerpt_hover_color',
            [
                'label'     => __('Excerpt Color', 'fastland-hp'),
                'type'      => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .fastland-service-widget-item:hover p' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->end_controls_tab();
        $this->end_controls_tabs();
        $this->end_controls_section();
        $this->start_controls_section(
            'button_style',
            [
                'label' => __('Button', 'fastland-hp'),
                'tab'   => \Elementor\Controls_Manager::TAB_STYLE, 'condition' => [
                    'show_readmore' => 'yes',
                ],
            ]
        );
        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name'     => 'btn_typography',
                'label'    => __('Button Typography', 'fastland-hp'),
                'selector' => '{{WRAPPER}} .service-btn',
            ]
        );
        $this->start_controls_tabs(
            'button_style_tabs'
        );
        $this->start_controls_tab(
            'button_style_normal_tab',
            [
                'label' => __('Normal', 'fastland-hp'),
            ]
        );
        $this->add_control(
            'btn_icon_color',
            [
                'label'     => __('Icon Color', 'fastland-hp'),
                'type'      => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .service-btn .btn-icon'      => 'color: {{VALUE}}',
                    '{{WRAPPER}} .service-btn .btn-icon path' => 'stroke: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'icon_fill_color',
            [
                'label'      => __('Icon Fill Color', 'fastland-hp'),
                'type'       => \Elementor\Controls_Manager::COLOR,
                'selectors'  => [
                    '{{WRAPPER}} .service-btn .btn-icon path' => 'fill: {{VALUE}}',
                ],
                'conditions' => [
                    'relation' => 'or',
                    'terms'    => [
                        [
                            'name'     => 'icon[library]',
                            'operator' => '==',
                            'value'    => 'svg',
                        ],
                    ],
                ],
            ]
        );
        $this->add_control(
            'boxed_btn_color',
            [
                'label'     => __('Button Color', 'fastland-hp'),
                'type'      => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .service-btn' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'boxed_btn_background',
            [
                'label'     => __('Background Color', 'fastland-hp'),
                'type'      => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .service-btn' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        $this->add_group_control(
            \Elementor\Group_Control_Box_Shadow::get_type(),
            [
                'name'     => 'button_shadow',
                'label'    => __('Button Shadow', 'fastland-hp'),
                'selector' => '{{WRAPPER}} .service-btn',
            ]
        );
        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name'     => 'button_border',
                'label'    => __('Border', 'fastland-hp'),
                'selector' => '{{WRAPPER}} .service-btn',
            ]
        );
        $this->add_control(
            'btn_icon_size',
            [
                'label'      => __('Icon Size', 'fastland-hp'),
                'type'       => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', 'em', '%'],
                'range'      => [
                    'px' => [
                        'min'  => 0,
                        'max'  => 50,
                        'step' => 1,
                    ],
                    '%'  => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .service-btn .btn-icon'     => 'font-size: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .service-btn .btn-icon svg' => 'width: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        $this->add_control(
            'btn_icon_gap',
            [
                'label'      => __('Icon gap', 'fastland-hp'),
                'type'       => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', 'em', '%'],
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                    '%'  => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .service-btn .icon-before, body.rtl {{WRAPPER}} .service-btn .icon-after '  => 'margin-right: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .service-btn .icon-after , body.rtl  {{WRAPPER}} .service-btn .icon-before' => 'margin-left: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        $this->add_responsive_control(
            'button_radius',
            [
                'label'      => __('Border Radius', 'fastland-hp'),
                'type'       => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .service-btn'          => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    'body.rtl {{WRAPPER}} .service-btn' => 'border-radius: {{TOP}}{{UNIT}} {{LEFT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{RIGHT}}{{UNIT}};',
                ],
            ]
        );
        $this->add_control(
            'buton_style_divider',
            [
                'type' => \Elementor\Controls_Manager::DIVIDER,
            ]
        );
        $this->add_responsive_control(
            'button_padding',
            [
                'label'      => __('Button Padding', 'fastland-hp'),
                'type'       => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .service-btn'          => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    'body.rtl {{WRAPPER}} .service-btn' => 'padding: {{TOP}}{{UNIT}} {{LEFT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{RIGHT}}{{UNIT}};',
                ],
            ]
        );
        $this->end_controls_tab();
        $this->start_controls_tab(
            'button_style_hover_tab',
            [
                'label' => __('Hover', 'fastland-hp'),
            ]
        );
        $this->add_control(
            'btn_icon_hover_color',
            [
                'label'     => __('Icon Color', 'fastland-hp'),
                'type'      => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .fastland-service-widget-item:hover .service-btn .btn-icon'      => 'color: {{VALUE}}',
                    '{{WRAPPER}} .fastland-service-widget-item:hover .service-btn .btn-icon path' => 'stroke: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'btn_icon_fill_color_hover',
            [
                'label'      => __('Icon Fill Color', 'fastland-hp'),
                'type'       => \Elementor\Controls_Manager::COLOR,
                'selectors'  => [
                    '{{WRAPPER}} .fastland-service-widget-item:hover .service-btn .btn-icon path' => 'fill: {{VALUE}}',
                ],
                'conditions' => [
                    'relation' => 'or',
                    'terms'    => [
                        [
                            'name'     => 'icon[library]',
                            'operator' => '==',
                            'value'    => 'svg',
                        ],
                    ],
                ],
            ]
        );
        $this->add_control(
            'btn_hover_color',
            [
                'label'     => __('Button Color', 'fastland-hp'),
                'type'      => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .fastland-service-widget-item:hover .service-btn' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'btn_hover_background',
            [
                'label'     => __('Background Color', 'fastland-hp'),
                'type'      => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .service-btn:hover' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name'     => 'button_hover_border',
                'label'    => __('Border', 'fastland-hp'),
                'selector' => '{{WRAPPER}} .service-btn:hover',
            ]
        );
        $this->add_control(
            'btn_hover_animation',
            [
                'label' => __('Hover Animation', 'fastland-hp'),
                'type'  => \Elementor\Controls_Manager::HOVER_ANIMATION,
            ]
        );
        $this->add_group_control(
            \Elementor\Group_Control_Box_Shadow::get_type(),
            [
                'name'     => 'button_hover_shadow',
                'label'    => __('Button Shadow', 'fastland-hp'),
                'selector' => '{{WRAPPER}} .service-btn:hover',
            ]
        );
        $this->add_responsive_control(
            'button_hover_radius',
            [
                'label'      => __('Border Radius', 'fastland-hp'),
                'type'       => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .service-btn:hover'          => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    'body.rtl {{WRAPPER}} .service-btn:hover' => 'border-radius: {{TOP}}{{UNIT}} {{LEFT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{RIGHT}}{{UNIT}};',
                ],
            ]
        );
        $this->add_control(
            'icon_hover_gap',
            [
                'label'      => __('Icon gap', 'fastland-hp'),
                'type'       => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', 'em', '%'],
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                    '%'  => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .service-btn:hover .icon-before' => 'transform: translatex( -{{SIZE}}{{UNIT}} );',
                    '{{WRAPPER}} .service-btn:hover .icon-after ' => 'transform: translatex( {{SIZE}}{{UNIT}} );',
                ],
            ]
        );
        $this->end_controls_tab();
        $this->end_controls_tabs();
        $this->end_controls_section();
        $this->start_controls_section(
            'section_content_box_style',
            [
                'label' => __('Box', 'fastland-hp'),
                'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );
        $this->start_controls_tabs(
            'box_style_tabs'
        );
        $this->start_controls_tab(
            'box_style_normal_tab',
            [
                'label' => __('Normal', 'fastland-hp'),
            ]
        );
        $this->add_control(
            'box_bg_color',
            [
                'label'     => __('Box Backgroound Color', 'fastland-hp'),
                'type'      => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .fastland-service-widget-item' => 'background-color: {{VALUE}};',
                ],
            ]
        );
        $this->add_group_control(
            \Elementor\Group_Control_Box_Shadow::get_type(),
            [
                'name'     => 'box_shadow',
                'label'    => __('Box Hover Shadow', 'fastland-hp'),
                'selector' => '{{WRAPPER}} .fastland-service-widget-item',
            ]
        );
        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name'     => 'box_border',
                'label'    => __('Box Border', ''),
                'selector' => '{{WRAPPER}} .fastland-service-widget-item',
            ]
        );
        $this->add_responsive_control(
            'box_radius',
            [
                'label'      => __('Box Radius', 'fastland-hp'),
                'type'       => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .fastland-service-widget-item' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        $this->add_responsive_control(
            'box_padding',
            [
                'label'      => __('Box Padding', 'fastland-hp'),
                'type'       => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .fastland-service-widget-item ' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        $this->end_controls_tab();
        $this->start_controls_tab(
            'box_style_hover_tab',
            [
                'label' => __('Hover', 'fastland-hp'),
            ]
        );
        $this->add_control(
            'box_hover_bg_color',
            [
                'label'     => __('Box Backgroound Color', 'fastland-hp'),
                'type'      => \Elementor\Controls_Manager::COLOR,
                'defautl'   => '#233aff',
                'selectors' => [
                    '{{WRAPPER}} .fastland-service-widget-item:hover' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        $this->add_responsive_control(
            'box_hover_radius',
            [
                'label'      => __('Box Radius', 'fastland-hp'),
                'type'       => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .fastland-service-widget-item:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        $this->add_group_control(
            \Elementor\Group_Control_Box_Shadow::get_type(),
            [
                'name'     => 'box_hover_shadow',
                'label'    => __('Box Hover Shadow', 'fastland-hp'),
                'selector' => '{{WRAPPER}} .fastland-service-widget-item:hover',
            ]
        );
        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name'     => 'box_hover_border',
                'label'    => __('Box Border', ''),
                'selector' => '{{WRAPPER}} .fastland-service-widget-item:hover ',
            ]
        );
        $this->end_controls_tab();
        $this->end_controls_tabs();
        $this->end_controls_section();
    }
    /**
     * Render the widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function render()
    {
        $settings           = $this->get_settings();
        $post_grid_desktop  = $settings['post_grid'];
        $post_grid_tablet   = isset($settings['post_grid_tablet']) ? $settings['post_grid_tablet'] : '';
        $post_grid_mobile   = isset($settings['post_grid_mobile']) ? $settings['post_grid_mobile'] : '';
        $post_grid          = sprintf('col-lg-%s col-md-%s col-%s', esc_attr($post_grid_desktop), esc_attr($post_grid_tablet), esc_attr($post_grid_mobile));
        $custom_css         = '';
        $paged              = get_query_var('paged') ? get_query_var('paged') : 1;
        $include_categories = array();
        $exclude_tags       = array();
        $include_tags       = array();
        $include_authors    = array();
        $exclude_categories = array();
        $exclude_authors    = array();
        $is_include_cat     = in_array('category', $settings['include_by']);
        $is_include_tag     = in_array('tags', $settings['include_by']);
        $is_include_author  = in_array('author', $settings['include_by']);
        $is_exclude_cat     = in_array('category', $settings['exclude_by']);
        $is_exclude_tag     = in_array('tags', $settings['exclude_by']);
        $is_exclude_author  = in_array('author', $settings['exclude_by']);
        $current_post_id    = '';

        if (0 != count($settings['include_categories'])) {
            $include_categories['tax_query'] = [
                'taxonomy' => 'service-category',
                'field'    => 'slug',
                'terms'    => $settings['include_categories'],
            ];
        }
        if (0 != count($settings['include_tags'])) {
            $include_tags = implode(',', $settings['include_tags']);
        }
        if (0 != count($settings['include_authors'])) {
            $include_authors = implode(',', $settings['include_authors']);
        }
        if (0 != count($settings['exclude_categories'])) {
            $exclude_categories['tax_query'] = [
                'taxonomy' => 'service-category',
                'operator' => 'NOT IN',
                'field'    => 'slug',
                'terms'    => $settings['exclude_categories'],
            ];
        }
        if (0 != count($settings['exclude_tags'])) {
            $exclude_tags['tax_query'] = [
                'taxonomy' => 'service-tag',
                'operator' => 'NOT IN',
                'field'    => 'slug',
                'terms'    => $settings['exclude_tags'],
            ];
        }
        if (0 != count($settings['exclude_authors'])) {
            $exclude_authors = implode(',', $settings['exclude_authors']);
        }
        if (in_array('current_post', $settings['exclude_by']) && is_single() && 'portfolio' == get_post_type()) {
            $current_post_id = get_the_ID();
        }
        // var_dump($settings['exclude_categories']);
        if ('related' == $settings['source'] && is_single() && 'portfolio' == get_post_type()) {
            $related_categories = get_the_terms(get_the_ID(), 'service-category');
            $related_cats       = [];
            if ($related_categories) {
                foreach ($related_categories as $related_cat) {
                    $related_cats[] = $related_cat->slug;
                }
            }
            $the_query = new WP_Query(array(
                'posts_per_page' => $settings['posts_per_page'],
                'post_type'      => 'service',
                'orderby'        => $settings['orderby'],
                'order'          => $settings['order'],
                'post__not_in'   => array($current_post_id),
                'paged'          => $paged,
                'tax_query'      => array(
                    array(
                        'taxonomy' => 'service-category',
                        'operator' => 'IN',
                        'field'    => 'slug',
                        'terms'    => $related_cats,
                    ),
                ),
            ));
        } elseif ('manual_selection' == $settings['source']) {
            $the_query = new WP_Query(array(
                'posts_per_page' => $settings['posts_per_page'],
                'post_type'      => 'service',
                'orderby'        => $settings['orderby'],
                'order'          => $settings['order'],
                'paged'          => $paged,
                'post__in'       => (0 != count($settings['manual_selection'])) ? $settings['manual_selection'] : array(),
            ));
        } else {
            $the_query = new WP_Query(array(
                'posts_per_page' => $settings['posts_per_page'],
                'post_type'      => 'service',
                'orderby'        => $settings['orderby'],
                'order'          => $settings['order'],
                'paged'          => $paged,
                'service-tag'    => ($is_include_tag && 0 != count($settings['include_tags'])) ? $include_tags : '',
                'post__not_in'   => array($current_post_id),
                'author'         => ($is_include_author && 0 != count($settings['include_authors'])) ? $include_authors : '',
                'author__not_in' => ($is_exclude_author && 0 != count($settings['exclude_authors'])) ? $exclude_authors : '',
                'tax_query'      => array(
                    'relation' => 'AND',
                    ($is_exclude_tag && 0 != count($settings['exclude_tags'])) ? $exclude_tags : '',
                    ($is_exclude_cat && 0 != count($settings['exclude_categories'])) ? $exclude_categories : '',
                    ($is_include_cat && 0 != count($settings['include_categories'])) ? $include_categories : '',
                ),
            ));
        }
?>
        <?php if ($the_query->have_posts()) : ?>
            <div class="row justify-content-center">
                <?php while ($the_query->have_posts()) : $the_query->the_post(); ?>
                    <?php
                    $idd       = get_the_ID();
                    $excerpt   = ($settings['excerpt_limit']['size']) ? wp_trim_words(get_the_excerpt(), $settings['excerpt_limit']['size'], '...') : get_the_excerpt();
                    $title     = ($settings['title_limit']['size']) ? wp_trim_words(get_the_title(), $settings['title_limit']['size'], '...') : get_the_title();
                    $icon      = get_field('icon');
                    $bg_color  = get_field('background_color');
                    $unq_class = 'fastland-service-' . $idd;
                    $custom_css .= "
			                .{$unq_class} .service-icon{
			                   background-color: {$bg_color};
			                   box-shadow: 0px 20px 20px {$bg_color}33;',
			                }
			                .{$unq_class} .service-btn:hover{
			                    color: {$bg_color};
			                }
			        ";
                    ?>
                    <div class="<?php printf('%s %s', $post_grid, $unq_class); ?> fastland-service-widget-wrap">
                        <div class="fastland-service-widget-item ">
                            <div class="service-icon-wrapper">
                                <div class="service-icon">
                                    <?php echo $icon; ?>
                                </div>
                            </div>
                            <div class="service-content-wrap">
                                <div class="service-content">
                                    <?php
                                    printf('<a href="%s" class="service-title-link d-block"><h3 class="service-title">%s</h3></a>', get_the_permalink(), $title);
                                    echo ('yes' == $settings['show_excerpt']) ? sprintf('<p> %s </p>', esc_html($excerpt)) : '';
                                    ?>
                                </div>
                                <?php if ('yes' == $settings['show_readmore']) : ?>
                                    <div class="service-btn-wrap">
                                        <a class="service-btn <?php echo esc_attr('elementor-animation-' . $settings['btn_hover_animation']) ?>" href="<?php the_permalink() ?>">
                                            <?php if ('before' == $settings['icon_position'] && !empty($settings['icon']['value'])) : ?>
                                                <span class="icon-before btn-icon"><?php \Elementor\Icons_Manager::render_icon($settings['icon'], ['aria-hidden' => 'true']) ?></span>
                                            <?php endif; ?>
                                            <?php echo esc_html($settings['readmore_text']); ?>
                                            <?php if ('after' == $settings['icon_position'] && !empty($settings['icon']['value'])) : ?>
                                                <span class="icon-after btn-icon"><?php \Elementor\Icons_Manager::render_icon($settings['icon'], ['aria-hidden' => 'true']) ?></span>
                                            <?php endif; ?>
                                        </a>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                <?php
                endwhile;
                wp_reset_postdata(); ?>
            </div>
<?php
        endif;
        printf('<style>%s</style>', esc_attr($custom_css));
    }
}
$widgets_manager->register(new \Fastland_Services());
