<?php
namespace Elementor;

if ( ! defined('ABSPATH') ) {
    exit; // Exit if accessed directly.
}

/**
 * Classic Elementor Addon Cea Draw Svg
 *
 * @since 1.0.0
 */
class CEA_Elementor_Draw_Svg_Widget extends \Elementor\Widget_Base {

    /**
	 * Get widget name.
	 *
	 * Retrieve Cea Draw Svg widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
    public function get_name(): string {
        return "ceadrawsvg";
    }

    /**
	 * Get widget title.
	 *
	 * Retrieve Cea Draw Svg widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title(): string {
        return esc_html__( "Draw SVG", 'classic-elementor-addons-pro' );
    }

    /**
	 * Get widget icon.
	 *
	 * Retrieve Cea Draw Svg widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon(): string {
        return "eicon-image-rollover";
    }

    /**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the Cea Draw Svg widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories(): array {
        return [ "classic-elements" ];
    }

    /**
     * Get Keywords
     *
     * Retrieve the list of keywords that used to search for Cea Draw SVG widget
     * 
     * @access public
     * 
     * @return array Widget Keywords
     */
	public function get_keywords(): array {
        return [ 'zozo', 'themes', 'draw', 'svg' ];
    }

    /**
	 * Retrieve the list of scripts the counter widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.3.0
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'cea-custom-front'  ];
	}

    /**
     * Retrieve the list of styles the counter widget depends on.
     *
     * Used to set styles dependencies required to run the widget.
     * 
     * @since 1.3.0
     * @access public
     * 
     * @return array Widget styles dependencies
     */
    public function get_style_depends(): array {
        return [ 'cea-editor' ];
    }

    /**
	 * Register Cea Draw Svg widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls(): void {

        // General Section
        $this->start_controls_section(
            'general',
            [
                'label' => esc_html__( 'General', 'classic-elementor-addons-pro' ),
                'tab'   => \Elementor\Controls_Manager::TAB_CONTENT
            ]
        );
        $this->add_control(
            'svg-src',
            [
                'label'     => esc_html__( 'Source', 'classic-elementor-addons-pro' ),
                'type'      => \Elementor\Controls_Manager::SELECT,
                'options'   => [
                    'icon'          => esc_html__( 'Icons', 'classic-elementor-addons-pro' ),
                    'custom_svg'    => esc_html__( 'Custom SVG', 'classic-elementor-addons-pro' )
                ],
                'default'   => 'icon'
            ]
        );
        $this->add_control(
            'icons',
            [
                'label'     => esc_html__( 'Icon', 'classic-elementor-addons-pro' ),
                'type'      => \Elementor\Controls_Manager::ICONS,
                'default'   => [
                    'value'     => 'fa-brands fa-java',
                    'library'   => 'fa-brands'
                ],
                'condition' => [
                    'svg-src'   => 'icon'
                ]
            ]
        );
        $this->add_control(
            'svg-code',
            [
                'label'     => esc_html__( 'SVG Code', 'classic-elementor-addons-pro' ),
                'type'      => \Elementor\Controls_Manager::CODE,
                'rows'      => 20,
                'ai'        => [
                    'active'    => false
                ],
                'default'   => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 384 512"><!--!Font Awesome Free 6.7.2 by @fontawesome - https://fontawesome.com License - https://fontawesome.com/license/free Copyright 2025 Fonticons, Inc.--><path d="M277.7 312.9c9.8-6.7 23.4-12.5 23.4-12.5s-38.7 7-77.2 10.2c-47.1 3.9-97.7 4.7-123.1 1.3-60.1-8 33-30.1 33-30.1s-36.1-2.4-80.6 19c-52.5 25.4 130 37 224.5 12.1zm-85.4-32.1c-19-42.7-83.1-80.2 0-145.8C296 53.2 242.8 0 242.8 0c21.5 84.5-75.6 110.1-110.7 162.6-23.9 35.9 11.7 74.4 60.2 118.2zm114.6-176.2c.1 0-175.2 43.8-91.5 140.2 24.7 28.4-6.5 54-6.5 54s62.7-32.4 33.9-72.9c-26.9-37.8-47.5-56.6 64.1-121.3zm-6.1 270.5a12.2 12.2 0 0 1 -2 2.6c128.3-33.7 81.1-118.9 19.8-97.3a17.3 17.3 0 0 0 -8.2 6.3 70.5 70.5 0 0 1 11-3c31-6.5 75.5 41.5-20.6 91.4zM348 437.4s14.5 11.9-15.9 21.2c-57.9 17.5-240.8 22.8-291.6 .7-18.3-7.9 16-19 26.8-21.3 11.2-2.4 17.7-2 17.7-2-20.3-14.3-131.3 28.1-56.4 40.2C232.8 509.4 401 461.3 348 437.4zM124.4 396c-78.7 22 47.9 67.4 148.1 24.5a185.9 185.9 0 0 1 -28.2-13.8c-44.7 8.5-65.4 9.1-106 4.5-33.5-3.8-13.9-15.2-13.9-15.2zm179.8 97.2c-78.7 14.8-175.8 13.1-233.3 3.6 0-.1 11.8 9.7 72.4 13.6 92.2 5.9 233.8-3.3 237.1-46.9 0 0-6.4 16.5-76.2 29.7zM260.6 353c-59.2 11.4-93.5 11.1-136.8 6.6-33.5-3.5-11.6-19.7-11.6-19.7-86.8 28.8 48.2 61.4 169.5 25.9a60.4 60.4 0 0 1 -21.1-12.8z"/></svg>',
                'condition' => [
                    'svg-src'   => 'custom_svg'
                ]
            ]
        );
        $this->add_responsive_control(
            'svg-width',
            [
                'label'         => esc_html__( 'Width', 'classic-elementor-addons-pro' ),
                'type'          => \Elementor\Controls_Manager::SLIDER,
                'devices'       => [ 'desktop', 'tablet', 'mobile' ],
                'size_units'    => [ 'px', '%' ],
                'range'         => [
                    'px'    => [
                        'min' => 100,
                        'max' => 300
                    ],
                    '%'     => [
                        'min' => 0,
                        'max' => 100
                    ]
                ],
                'default'       => [
                    'unit'  => 'px',
                    'size'  => 200
                ],
                'tablet_default' => [
                    'unit' => 'px',
                    'size' => 160
                ],
                'mobile_default' => [
                    'unit' => 'px',
                    'size' => 120
                ],
                'selectors' => [
                    '{{WRAPPER}} .cea-draw-svg-widget span svg' => 'width: {{SIZE}}{{UNIT}} !important;'
                ]
            ]
        );
        $this->add_control(
            'svg-align',
            [
                'label' => esc_html__( 'Alignment', 'classic-elementor-addons-pro' ),
                'type'  => \Elementor\Controls_Manager::CHOOSE,
				'options' => [
					'left' => [
						'title' => esc_html__( 'Left', 'classic-elementor-addons-pro' ),
						'icon' => 'eicon-text-align-left',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'classic-elementor-addons-pro' ),
						'icon' => 'eicon-text-align-center',
					],
					'right' => [
						'title' => esc_html__( 'Right', 'classic-elementor-addons-pro' ),
						'icon' => 'eicon-text-align-right',
					],
				],
                'default' => 'center',
                'selectors' => [
                    '{{WRAPPER}} .cea-draw-svg-widget' => 'justify-content: {{VALUE}};'
                ]
            ]
        );
        $this->end_controls_section();

        // Animation Section
        $this->start_controls_section(
            'animation',
            [
                'label' => esc_html__( 'Animation', 'classic-elementor-addons-pro' ),
                'tab'   => \Elementor\Controls_Manager::TAB_CONTENT
            ]
        );
        $this->add_control(
            'anime-load',
            [
                'label' => esc_html__( 'Animation', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::SELECT,
                'options' => [
                    'load' => esc_html__( 'On Page Load', 'classic-elementor-addons-pro' ),
                    'paused' => esc_html__( 'On Mouse Hover', 'classic-elementor-addons-pro' ),
                    'running' => esc_html__( 'Stop on Hover', 'classic-elementor-addons-pro' ),
                    'scrolled' => esc_html__( 'On Page Scroll', 'classic-elementor-addons-pro' )
                ],
                'default' => 'load'
            ]
        );
        $this->add_control(
            'anime-duration',
            [
                'label' => esc_html__( 'Animation Duration', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => [ 's', 'ms' ],
                'range' => [
                    's' => [
                        'min' => 0,
                        'max' => 10
                    ],
                    'ms' => [
                        'min' => 0,
                        'max' => 1000
                    ]
                ],
                'default' => [
                    'unit' => 's',
                    'size' => 5
                ],
                'selectors' => [
                    '{{WRAPPER}} .cea-draw-svg-widget span svg path' => 'animation-duration: {{SIZE}}{{UNIT}};'
                ]
            ]
        );
        $this->add_control(
            'anime-delay',
            [
                'label' => esc_html__( 'Animation Delay', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'description' => 'In terms of seconds',
                'min' => 0,
                'max' => 10,
                'step' => 1,
                'default' => 0,
                'selectors' => [
                    '{{WRAPPER}} .cea-draw-svg-widget span svg path' => 'animation-delay: {{VALUE}}s;'
                ]
            ]
        );
        $this->add_control(
            'anime-time',
            [
                'label' => esc_html__( 'Animation Timing Function', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::SELECT,
                'options' => [
                    'linear' => esc_html__( 'Linear', 'classic-elementor-addons-pro' ),
                    'ease' => esc_html__( 'Ease', 'classic-elementor-addons-pro' ),
                    'ease-in' => esc_html__( 'Ease-In', 'classic-elementor-addons-pro' ),
                    'ease-out' => esc_html__( 'Ease-Out', 'classic-elementor-addons-pro' ),
                    'ease-in-out' => esc_html__( 'Ease-In-Out', 'classic-elementor-addons-pro' )
                ],
                'default' => 'linear',
                'selectors' => [
                    '{{WRAPPER}} .cea-draw-svg-widget span svg path' => 'animation-timing-function: {{VALUE}};'
                ]
            ]
        );
        $this->add_control(
            'anime-iterate',
            [
                'label' => esc_html__( 'Repeat Drawing', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::SELECT,
                'options' => [
                    'infinite' => esc_html__( 'Yes', 'classic-elementor-addons-pro' ),
                    1 => esc_html__( 'No', 'classic-elementor-addons-pro' )
                ],
                'default' => 'infinite',
                'selectors' => [
                    '{{WRAPPER}} .cea-draw-svg-widget span svg path' => 'animation-iteration-count: {{VALUE}};'
                ]
            ]
        );
        $this->end_controls_section();

        // $this->start_controls_section(
        //     'responsive',
        //     [
        //         'label' => esc_html__( 'Responsive', 'classic-elementor-addons-pro' ),
        //         'tab' => \Elementor\Controls_Manager::TAB_CONTENT
        //     ]
        // );

        // $this->add_control(
        //     'hide_on_lap',
        //     [
        //         'label' => esc_html__( 'Hide on Desktop', 'classic-elementor-addons-pro' ),
        //         'type' => \Elementor\Controls_Manager::SWITCHER,
        //         'return_value' => 'yes',
        //         'default' => 'no'
        //     ]
        // );

        // $this->add_control(
        //     'hide_on_lap',
        //     [
        //         'label' => esc_html__( 'Hide on Desktop', 'classic-elementor-addons-pro' ),
        //         'type' => \Elementor\Controls_Manager::SWITCHER,
        //         'return_value' => 'yes',
        //         'default' => 'no'
        //     ]
        // );

        // $this->add_control(
        //     'hide_on_lap',
        //     [
        //         'label' => esc_html__( 'Hide on Desktop', 'classic-elementor-addons-pro' ),
        //         'type' => \Elementor\Controls_Manager::SWITCHER,
        //         'return_value' => 'yes',
        //         'default' => 'no'
        //     ]
        // );

        // $this->end_controls_section();

        // Style Section
        $this->start_controls_section(
            'style',
            [
                'label' => esc_html__( 'Styles', 'classic-elementor-addons-pro' ),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE
            ]
        );
        $this->add_responsive_control(
			'margin_svg',
			[
				'label' => esc_html__( 'Margin', 'classic-elementor-addons-pro' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors' => [
					'{{WRAPPER}} .cea-draw-svg-widget span' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				]
			]
		);
        $this->add_responsive_control(
			'padding_svg',
			[
				'label' => esc_html__( 'Padding', 'classic-elementor-addons-pro' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors' => [
					'{{WRAPPER}} .cea-draw-svg-widget span' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				]
			]
		);
        $this->add_control(
            'border_want',
            [
                'label' => esc_html__( 'Border Display', 'classic-elementor-addons-pro' ),
                'type'  => \Elementor\Controls_Manager::SWITCHER,
                'return_value' => 'yes',
                'default' => 'no'
            ]
        );
        $this->add_control(
            'border-color',
            [
                'label' => esc_html__( 'Border Color', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::COLOR,
                'condition' => [
                    'border_want' => 'yes'
                ],
                'default' => '#000000',
                'selectors' => [
                    '{{WRAPPER}} .cea-draw-svg-widget span' => 'border-color: {{VALUE}}'
                ]
            ]
        );
        $this->add_control(
            'border-width',
            [
                'label' => esc_html__( 'Border Width', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'condition' => [
                    'border_want' => 'yes'
                ],
                'range' => [
                    'min' => 0,
                    'max' => 50
                ],
                'default' => [
                    'size' => 1
                ],
                'selectors' => [
                    '{{WRAPPER}} .cea-draw-svg-widget span' => 'border-width: {{SIZE}}px;'
                ]
            ]
        );
        $this->add_control(
            'border-style',
            [
                'label' => esc_html__( 'Border Style', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::SELECT,
                'options' => [
                    'dashed' => esc_html__( 'Dashed', 'classic-elementor-addons-pro' ), 
                    'dotted' => esc_html__( 'Dotted', 'classic-elementor-addons-pro' ), 
                    'double' => esc_html__( 'Double', 'classic-elementor-addons-pro' ), 
                    'groove' => esc_html__( 'Groove', 'classic-elementor-addons-pro' ), 
                    'hidden' => esc_html__( 'Hidden', 'classic-elementor-addons-pro' ), 
                    'inset' => esc_html__( 'Inset', 'classic-elementor-addons-pro' ), 
                    'none' => esc_html__( 'None', 'classic-elementor-addons-pro' ), 
                    'outset' => esc_html__( 'Outset', 'classic-elementor-addons-pro' ), 
                    'ridge' => esc_html__( 'Ridge', 'classic-elementor-addons-pro' ), 
                    'solid' => esc_html__( 'Solid', 'classic-elementor-addons-pro' ), 
                ],
                'default' => 'solid',
                'condition' => [
                    'border_want' => 'yes'
                ],
                'selectors' => [
                    '{{WRAPPER}} .cea-draw-svg-widget span' => 'border-style: {{VALUE}};'
                ]
            ]
            
        );
        $this->add_responsive_control(
			'border-radius',
			[
				'label' => esc_html__( 'Border Radius', 'classic-elementor-addons-pro' ),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
                'condition' => [
                    'border_want' => 'yes'
                ],
				'selectors' => [
					'{{WRAPPER}} .cea-draw-svg-widget span' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				]
			]
		);
        $this->add_control(
            'bg-color',
            [
                'label' => esc_html__( 'Background Color', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .cea-draw-svg-widget span' => 'background-color: {{VALUE}};'
                ]
            ]
        );
        $this->add_control(
            'color',
            [
                'label' => esc_html__( 'Stroke Color', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#974CF3',
                'selectors' => [
                    '{{WRAPPER}} .cea-draw-svg-widget span svg path' => 'stroke: {{VALUE}};'
                ]
            ]
        );
        $this->add_control(
            'stroke-width',
            [
                'label' => esc_html__( 'Stroke Width', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'range' => [
                    'min' => 0,
                    'max' => 100
                ],
                'default' => [
                    'size' => 10
                ],
                'selectors' => [
                    '{{WRAPPER}} .cea-draw-svg-widget span svg path' => 'stroke-width: {{SIZE}};'
                ]
            ]
        );
        $this->add_control(
            'fill-now',
            [
                'label' => esc_html__( 'Fill', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'return_value' => 'yes',
                'default' => 'no'
            ]
        );
        $this->add_control(
            'fill-color',
            [
                'label' => esc_html__( 'Fill Color', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::COLOR,
                'condition' => [
                    'fill-now' => 'yes'
                ],
                'selectors' => [
                    '{{WRAPPER}} .cea-draw-svg-widget span svg path' => 'fill: {{VALUE}};'
                ]
            ]
        );
        $this->end_controls_section();
    }

    /**
	 * Render Tab widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render(): void {

        $settings = $this->get_settings_for_display();
        ?>
        <div class="cea-draw-svg-widget">
            <?php if( $settings['icons'] ): ?>
                <span class="<?php _e($settings['anime-load']); ?>"><?php \Elementor\Icons_Manager::render_icon( $settings['icons'], [ 'aria-hidden' => 'true' ] ); ?></span>
            <?php else: ?>
                <span class="<?php _e($settings['anime-load']); ?>"><?php _e($settings['svg-code']); ?></span>
            <?php endif; ?>
        </div>
        <?php
    }

}

