<?php

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Classic Elementor Addon List Step
 *
 * @since 1.0.0
 */
class CEA_Elementor_List_Step_Widget extends \Elementor\Widget_Base {

    /**
	 * Get widget name.
	 *
	 * Retrieve Cea List Step widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return "cealiststep";
	}

    /**
	 * Get widget title.
	 *
	 * Retrieve Cea List Step widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( "List Step", 'classic-elementor-addons-pro' );
	}

    /**
	 * Get widget icon.
	 *
	 * Retrieve Cea List Step widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return "ti-view-list";
	}

    /**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the Cea List Step widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories(): array {
        return [ "classic-elements" ];
    }

    /**
     * Get Keywords
     *
     * Retrieve the list of keywords that used to search for Cea List Step widget
     * 
     * @access public
     * 
     * @return array Widget Keywords
     */
    public function get_keywords(): array {
        return [ 'zozo', 'themes', 'image', 'zoom' ];
    }

    /**
	 * Retrieve the list of scripts the counter widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.3.0
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'cea-custom-front'  ];
	}

    /**
     * Retrieve the list of styles the counter widget depends on.
     *
     * Used to set styles dependencies required to run the widget.
     * 
     * @since 1.3.0
     * @access public
     * 
     * @return array Widget styles dependencies
     */
    public function get_style_depends(): array {
        return [ 'cea-editor' ];
    }

    /**
	 * Register Cea List Step widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
    protected function register_controls(): void {

        // Content Tab Starts
        $this->start_controls_section(
			"general_content",
			[
				"label"	=> esc_html__( "General", 'classic-elementor-addons-pro' ),
				"tab"	=> Controls_Manager::TAB_CONTENT,
            ]
        );
        $this->add_control(
            'title_main',
            [
                'label' => esc_html__( 'Title', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::TEXT,
                'label_block' => true,
                'default' => 'This is Title Text',
                'separator' => 'after',
            ]
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'list_title',
            [
                'label' => esc_html__( 'List Title', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::TEXT,
                'label_block' => true,
                'default' => __( 'This is List Title', 'classic-elementor-addons-pro' ),
            ]
        );
        $repeater->add_control(
            'list_content',
            [
                'label' => esc_html__( 'List Content', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::TEXTAREA,
                'default' => 'Lorem ipsum odor amet, consectetuer adipiscing elit. Nibh sapien per lectus sollicitudin sodales, sem suscipit augue. Luctus arcu maecenas donec donec venenatis sapien auctor est. Sapien ullamcorper tristique donec tempus libero ante porta fusce. Magna sodales curabitur hac nec suscipit malesuada consectetur. Diam enim accumsan fusce rhoncus fusce integer',
            ]
        );
        $repeater->add_control(
            'list_btn',
            [
                'label' => esc_html__( 'Enable Button?', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'return_value' => 'yes',
                'default' => 'yes'
            ]
        );
        $repeater->add_control(
            'btn_text',
            [
                'label' => esc_html__( 'Button Text', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => 'Click Here',
                'condition' => [
                    'list_btn' => 'yes'
                ]
            ]
        );
        $repeater->add_control(
            'btn_link',
            [
                'label' => esc_html__( 'Button Link', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::TEXT,
                'label_block' => true,
                'default' => '#',
                'condition' => [
                    'list_btn' => 'yes'
                ],
            ]
        );
        $repeater->add_control(
            'right_add',
            [
                'label' => esc_html__( 'Enable Image?', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::SELECT,
                'options' => [
                    'yes' => esc_html__( 'Yes', 'classic-elementor-addons-pro' ),
                    'no' => esc_html__( 'No', 'classic-elementor-addons-pro' ),
                ],
                'default' => 'yes',
            ]
        );
        $repeater->add_control(
            'list_image',
            [
                'label' => esc_html__( 'Image', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::MEDIA,
                'condition' => [
                    'right_add' => 'yes'
                ],
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
            ]
        );
        $repeater->add_control(
            'image_animation',
            [
                'label' => esc_html__( 'Image Animation', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::ANIMATION,
                'frontend_available' => true,
                'condition' => [
                    'right_add' => 'yes'
                ],
                'selectors' => [
                    '{{WRAPPER}} .list-step-image' => 'animation: {{VALUE}};',
                ]
            ]
        );

        $this->add_control(
            'repeater_list',
            [
                'label' => esc_html__( 'List Items', 'classic-elementor-addons-pro' ),
                'type'  => \Elementor\Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'list_title' => esc_html__( 'List Item 1', 'classic-elementor-addons-pro' ),
                    ],
                    [
                        'list_title' => esc_html__( 'List Item 2', 'classic-elementor-addons-pro' ),
                    ],
                    [
                        'list_title' => esc_html__( 'List Item 3', 'classic-elementor-addons-pro' ),
                    ],
                ],
                'title_field'	=> '{{{ list_title }}}',
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
            'general',
            [
                'label' => esc_html__( 'Layout', 'classic-elementor-addons-pro' ),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'list_padding',
            [
                'label' => esc_html__( 'Padding', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'rem', 'em' ],
                'selectors' => [
                    '{{WRAPPER}} .list-step-item' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'separator' => 'after',
            ]
        );

        $this->add_control(
            'list_clr',
            [
                'label' => esc_html__( 'Color', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .list-step-item' => 'color: {{VALUE}};'
                ]
            ]
        );

        $this->add_control(
            'list_back_clr',
            [
                'label' => esc_html__( 'Background Color', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .list-step-item' => 'background-color: {{VALUE}};'
                ]
            ]
        );

        $this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name' => 'list_item_shadow',
				'selector' => '{{WRAPPER}} .list-step-item',
			]
		);

        $this->end_controls_section();
        
        $this->start_controls_section(
            'title_style',
            [
                'label' => esc_html__( 'Title', 'classic-elementor-addons-pro' ),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'title_margin',
            [
                'label' => esc_html__( 'Margin', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'rem', 'em' ],
                'selectors' => [
                    '{{WRAPPER}} .list-step-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ]
            ]
        );

        $this->add_responsive_control(
            'title_padding',
            [
                'label' => esc_html__( 'Padding', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'rem', 'em' ],
                'selectors' => [
                    '{{WRAPPER}} .list-step-title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ]
            ]
        );

        $this->add_control(
            'title_color',
            [
                'label' => esc_html__( 'Color', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .list-step-title' => 'color: {{VALUE}};',
                ]
            ]
        );

        $this->add_responsive_control(
			'title_spacing',
			[
				'label' => esc_html__( 'Title Spacing', 'classic-elementor-addons-pro' ),
				'type' => Controls_Manager::SLIDER,
				'default' => [
					'size' => 5,
				],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .list-step-title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);

        $this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' 			=> 'title_typography',
				'selector' 		=> '{{WRAPPER}} .list-step-title',
			]
		);	

        $this->add_group_control(
			\Elementor\Group_Control_Text_Shadow::get_type(),
			[
				'name' => 'title_shadow',
				'selector' => '{{WRAPPER}} .list-step-title',
			]
		);

		$this->end_controls_section();

        $this->start_controls_section(
            'list_title',
            [
                'label' => esc_html__( 'List Title', 'classic-elementor-addons-pro' ),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
			'list_title_align',
			[
				'label' => esc_html__( 'Alignment', 'classic-elementor-addons-pro' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'left'    => [
						'title' => esc_html__( 'Left', 'classic-elementor-addons-pro' ),
						'icon' => 'eicon-text-align-left',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'classic-elementor-addons-pro' ),
						'icon' => 'eicon-text-align-center',
					],
					'right' => [
						'title' => esc_html__( 'Right', 'classic-elementor-addons-pro' ),
						'icon' => 'eicon-text-align-right',
					],
					'justify' => [
						'title' => esc_html__( 'Justified', 'classic-elementor-addons-pro' ),
						'icon' => 'eicon-text-align-justify',
					],
				],				
				'default' => 'justify',
				'selectors' => [
					'{{WRAPPER}} .list-step-head' => 'text-align: {{VALUE}};',
				],
			]
		);

        $this->add_responsive_control(
            'list_title_margin',
            [
                'label' => esc_html__( 'Margin', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'rem', 'em' ],
                'selectors' => [
                    '{{WRAPPER}} .list-step-head' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ]
            ]
        );

        $this->add_responsive_control(
            'list_title_padding',
            [
                'label' => esc_html__( 'Padding', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'rem', 'em' ],
                'selectors' => [
                    '{{WRAPPER}} .list-step-head' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ]
            ]
        );

        $this->add_control(
            'list_title_color',
            [
                'label' => esc_html__( 'Color', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .list-step-head' => 'color: {{VALUE}};',
                ]
            ]
        );

        $this->add_responsive_control(
			'list_title_spacing',
			[
				'label' => esc_html__( 'Title Spacing', 'classic-elementor-addons-pro' ),
				'type' => Controls_Manager::SLIDER,
				'default' => [
					'size' => 5,
				],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .list-step-head' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);

        $this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' 			=> 'list_title_typography',
				'selector' 		=> '{{WRAPPER}} .list-step-head',
			]
		);

        $this->end_controls_section();

        $this->start_controls_section(
            'list_para',
            [
                'label' => esc_html__( 'List Paragraph', 'classic-elementor-addons-pro' ),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
			'list_para_align',
			[
				'label' => esc_html__( 'Alignment', 'classic-elementor-addons-pro' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'left'    => [
						'title' => esc_html__( 'Left', 'classic-elementor-addons-pro' ),
						'icon' => 'eicon-text-align-left',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'classic-elementor-addons-pro' ),
						'icon' => 'eicon-text-align-center',
					],
					'right' => [
						'title' => esc_html__( 'Right', 'classic-elementor-addons-pro' ),
						'icon' => 'eicon-text-align-right',
					],
					'justify' => [
						'title' => esc_html__( 'Justified', 'classic-elementor-addons-pro' ),
						'icon' => 'eicon-text-align-justify',
					],
				],				
				'default' => 'left',
				'selectors' => [
					'{{WRAPPER}} .list-step-para' => 'text-align: {{VALUE}};',
				],
			]
		);

        $this->add_responsive_control(
            'list_para_margin',
            [
                'label' => esc_html__( 'Margin', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'rem', 'em' ],
                'selectors' => [
                    '{{WRAPPER}} .list-step-para' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ]
            ]
        );

        $this->add_responsive_control(
            'list_para_padding',
            [
                'label' => esc_html__( 'Padding', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'rem', 'em' ],
                'selectors' => [
                    '{{WRAPPER}} .list-step-para' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ]
            ]
        );

        $this->add_control(
            'list_para_color',
            [
                'label' => esc_html__( 'Color', 'classic-elementor-addons-pro' ),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .list-step-para' => 'color: {{VALUE}};',
                ]
            ]
        );

        $this->add_responsive_control(
			'list_para_spacing',
			[
				'label' => esc_html__( 'Para Spacing', 'classic-elementor-addons-pro' ),
				'type' => Controls_Manager::SLIDER,
				'default' => [
					'size' => 5,
				],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .list-step-para' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);

        $this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' 			=> 'list_title_typography',
				'selector' 		=> '{{WRAPPER}} .list-step-para',
			]
		);

        $this->end_controls_section();

        $this->start_controls_section(
            'list_button',
            [
                'label' => esc_html__( 'List Button', 'classic-elementor-addons-pro' ),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
			'button_align',
			[
				'label' => esc_html__( 'Alignment', 'classic-elementor-addons-pro' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'left'    => [
						'title' => esc_html__( 'Left', 'classic-elementor-addons-pro' ),
						'icon' => 'eicon-text-align-left',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'classic-elementor-addons-pro' ),
						'icon' => 'eicon-text-align-center',
					],
					'right' => [
						'title' => esc_html__( 'Right', 'classic-elementor-addons-pro' ),
						'icon' => 'eicon-text-align-right',
					],
					'justify' => [
						'title' => esc_html__( 'Justified', 'classic-elementor-addons-pro' ),
						'icon' => 'eicon-text-align-justify',
					],
				],
                'prefix_class' => 'cea-btn%s-align-',
				'default' => '',
			]
        );

        $this->add_responsive_control(
			'button_padding',
			[
				'label' => esc_html__( 'Padding', 'classic-elementor-addons-pro' ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors' => [
					'{{WRAPPER}} .list-step-btn' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

        $this->start_controls_tabs( 'button_clr' );

		$this->start_controls_tab(
			'button_normal',
			[
				'label' => esc_html__( 'Normal', 'classic-elementor-addons-pro' ),
			]
		);
		$this->add_control(
			'button_text_color',
			[
				'label' => esc_html__( 'Text Color', 'classic-elementor-addons-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .list-step-btn' => 'fill: {{VALUE}}; color: {{VALUE}};',
				],
			]
		);
		$this->add_control(
			'button_background_color',
			[
				'label' => esc_html__( 'Background Color', 'classic-elementor-addons-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .list-step-btn' => 'background-color: {{VALUE}};',
				],
			]
		);
		$this->end_controls_tab();

        $this->start_controls_tab(
            'button_hover',
            [
                'label' => esc_html__( 'Hover', 'classic-elementor-addons-pro' ),
            ]
        );
        $this->add_control(
			'btn_hover_clr',
			[
				'label' => esc_html__( 'Text Color', 'classic-elementor-addons-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .list-step-btn:hover' => 'fill: {{VALUE}}; color: {{VALUE}};',
				],
			]
		);
		$this->add_control(
			'btn_hover_back',
			[
				'label' => esc_html__( 'Background Color', 'classic-elementor-addons-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .list-step-btn:hover' => 'background-color: {{VALUE}};',
				],
			]
		);
		$this->end_controls_tab();

        $this->end_controls_tabs();

        $this->add_control(
            'btn_hover',
            [
                'label' => esc_html__( 'Hover Animation', 'classic-elementor-addons-pro' ),
                'type'  => \Elementor\Controls_Manager::HOVER_ANIMATION,
                'separator' => 'after',
            ]
        );

        $this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name' => 'border_btn',
				'selector' => '{{WRAPPER}} .list-step-btn',
				'separator' => 'before',
			]
		);

		$this->add_responsive_control(
			'button_border_radius',
			[
				'label' => esc_html__( 'Border Radius', 'classic-elementor-addons-pro' ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'selectors' => [
					'{{WRAPPER}} .list-step-btn' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

        $this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' 			=> 'btn_text_typo',
				'selector' 		=> '{{WRAPPER}} .list-step-para',
			]
		);

        $this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name' => 'button_box_shadow',
				'selector' => '{{WRAPPER}} .list-step-btn',
			]
		);

        $this->end_controls_section();

        $this->start_controls_section(
            'style_image',
            [
                'label' => esc_html__( 'List Image', 'classic-elementor-addons-pro' ),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
			'image_width',
			[
				'label' => esc_html__( 'Width', 'classic-elementor-addons-pro' ),
				'type' => Controls_Manager::SLIDER,
                'default' => [
                    'size' => 350,
                ],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 500,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .list-step-image' => 'width: {{SIZE}}{{UNIT}}; height: auto;',
				],
			]
		);

        $this->start_controls_tabs('list_image_style');
        $this->start_controls_tab( 
            'normal',
			[
				'label' => esc_html__( 'Normal', 'elementor' ),
			]
		);
		$this->add_control(
			'opacity',
			[
				'label' => esc_html__( 'Opacity', 'elementor' ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'max' => 1,
						'min' => 0.10,
						'step' => 0.01,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .list-step-image' => 'opacity: {{SIZE}};',
				],
			]
		);
		$this->add_group_control(
			Group_Control_Css_Filter::get_type(),
			[
				'name' => 'css_filters',
				'selector' => '{{WRAPPER}} .list-step-image',
			]
		);
        $this->end_controls_tab();
        $this->start_controls_tab( 
            'hover',
			[
				'label' => esc_html__( 'Hover', 'elementor' ),
			]
		);
		$this->add_control(
			'opacity_img_hover',
			[
				'label' => esc_html__( 'Opacity', 'elementor' ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'max' => 1,
						'min' => 0.10,
						'step' => 0.01,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .list-step-image:hover' => 'opacity: {{SIZE}};',
				],
			]
		);
		$this->add_group_control(
			Group_Control_Css_Filter::get_type(),
			[
				'name' => 'img_hover_filter',
				'selector' => '{{WRAPPER}} .list-step-image:hover',
			]
		);
        $this->end_controls_tab();
        $this->end_controls_tabs();

        $this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name' => 'border_img',
				'selector' => '{{WRAPPER}} .list-step-image img',
				'separator' => 'before',
			]
		);

		$this->add_responsive_control(
			'img_border_radius',
			[
				'label' => esc_html__( 'Border Radius', 'classic-elementor-addons-pro' ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'selectors' => [
					'{{WRAPPER}} .list-step-image img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

        $this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name' => 'image_box_shadow',
				'selector' => '{{WRAPPER}} .list-step-image img',
			]
		);

        $this->end_controls_section();

    }

    protected function render(): void {
        $settings = $this->get_settings_for_display();
        $title  = $settings['title_main']; 
        $button_hover = $settings['btn_hover'];
        
        echo '<div class="list-step">';
            echo '<h2 class="list-step-title text-center"> ' . $title . '</h2>';
            echo '<div class="list-step-container">';
                foreach( $settings['repeater_list'] as $list ) {
                    $list_title = $list['list_title'];
                    $list_text = $list['list_content'];
                    $apply_btn = $list['list_btn'];
                    $list_btn = $list['btn_text'] ?? 'Click Here';
                    $list_link = $list['btn_link'] ?? '#';
                    $right_apply = $list['right_add'];
                    $list_image = $list['list_image'];
                    $image_animation = $list['image_animation'];
                
                    echo '<div class="list-step-item">';
                    echo '<div class="list-step-content">';
                        echo '<div class="list-step-right">';
                            echo '<h4 class="list-step-head"> ' . $list_title . '</h4>';
                            echo '<p class="list-step-para">' . $list_text . '</p>';
                        if( $apply_btn == 'yes' ) {
                            echo '<div class="cea-button-wrapper">
                                    <a class="btn list-step-btn elementor-animation-' . $button_hover . '" href="' . $list_link . '" > ' . $list_btn . '</a>
                                  </div>';
                        }
                        echo '</div><!--div.list-step-right-->';
                    if( $right_apply != 'no' ) {
                        echo '<div class="list-step-left">';
                        if( $right_apply == 'yes' ) {
                            echo '<div class="list-step-image animated ' . $image_animation . '">';
                                echo '<img src="' . $list_image['url'] . '" alt="' . $list_title . '" />';
                            echo '</div><!--div.list-step-image-->';
                        }
                        echo '</div><!--div.list-step-left-->';
                    }
                    echo '</div><!--div.list-step-content-->';   
                    echo '</div><!--div.list-step-item-->';
                }
            echo '</div><!--div.list-step-container-->';
        echo '</div><!--div.list-step-->';
    }

}